<?php

namespace App\Utils\User;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

trait ManagesUser {

    public function loginAsUser(User $user) {
        auth('web')->loginUsingId($user->id);
        session()->put('login-approved', true);
        return to_route('dashboard')->with('success', "Logged in to $user->name's account.");
    }


    protected function getUserAndUserDetails(Request $request, $user_id) {
        $user = $request->only(
            'name',
            'email',
            'username',
            'account_number',
            'phone_number',
            'pin'
        );

        $userDetails = $request->only(
            'account_type',
            'date_of_birth',
            'address',
            'country',
            'gender',
            'marital_status',
            'occupation',
            'ssn',
        );

        if ($user_id) {
            $updatedUser = User::find($user_id);

            foreach ($updatedUser->details->codes as $code => $_) {
                if ($request->has($code)) {
                    $userDetails['codes'][$code] = $request->$code;
                }
            }

            foreach ($updatedUser->details->permissions as $permission => $_) {
                if ($request->has($permission)) {

                    $userDetails['permissions'][$permission] = !empty($request->$permission);
                }
            }
        } else {
            $userDetails['codes'] = [
                'tac' => random_int(10000, 99999),
                'tax' => random_int(10000, 99999),
            ];
            $userDetails['permissions'] = [
                'allow_transfer' =>  true,
                'validate_transfer' => true,
            ];
            $user['account_number'] = account_number();
            $user['password_text'] = $request->password;
            $user['password'] = Hash::make($request->password);
            $userDetails['currency'] = $request->currency;
        }

        return [$user, $userDetails];
    }

    public function uploadAvatar(Request $request) {
        $user = User::find(session('user_id'));
        $data = Validator::make(request()->all(), [
            'avatar' => 'max:3072|mimes:jpeg,jpg,png,gif|required',
        ]);

        if ($data->fails()) {
            return response()->json(['errors' => $data->errors()->all()]);
        } else {
            try {
                $avatar = '/uploads/' . $request->file('avatar')->store('avatar', 'storage');
                $user->details()->update([
                    'avatar' => $avatar
                ]);
                return response()->json(['success' => 'Profile Updated Successfully.']);
            } catch (\Throwable $th) {
                return response()->json(['info' => $th->getMessage()]);
            }
        }
    }

    protected function validationRules(): array {
        $user = User::find(session('user_id'));
        return [
            'name' => 'required|min:2|max:255',
            'phone_number' => ['required', 'string', Rule::unique('users')->ignore($user->id)],
            'email' => 'required|unique:users,email,' . $user->id,
            'username' => ['required', 'string', 'max:255', 'regex:/^\S*$/', 'alpha_num', Rule::unique('users')->ignore($user->id)],
            'pin' => 'required|numeric',
            'country' => 'required|string',
            'address' => 'required|string',
            'date_of_birth' => ['required', 'date', 'before_or_equal:' . now()->subYears(18)->format('Y-m-d')],
            'account_type' => 'required|string|max:255',
            'marital_status' => 'required|string|max:255',
            'occupation' => 'required|string',
            'currency' => 'required',
            'ssn' => 'required',
            'gender' => 'required'
        ];
    }

    protected function changeStatus(User $user, bool $is_active) {
        $user->update([
            'is_active' => $is_active,
        ]);
    }
}
