<?php

namespace App\Utils\User;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\TransactionLog;
use App\Utils\Enums\AlertType;
use App\Utils\Enums\TransactionType;
use App\Notifications\User\UserNotice;

trait ManagesTransfer {
    public function creditOrDebit(Request $request, int $account_number = null) {
        $request->validate([
            'alert_type' => 'required',
            'amount' => 'required|numeric',
            'note' => 'required',
            'date' => 'required'
        ]);

        $user = User::find(session('user_id'));
        
        if($account_number) {
            $user = User::where('account_number', $account_number)->first();
            
            if($user==null) {
                $error = 'User with this account number doesn\'t exist';
                return back()->with('error', $this->message($error))->withErrors(['account_number' => $error])->withInput();
            }
        }

        $transaction = $this->createTransaction($user, $request);

        switch ($request->alert_type) {
            case 'debit':
                if ($user->details->account_balance < $request->amount) {
                    return back()->with('success', $this->message('Amount to be debited is greater than account balance.'));
                } else {
                    $user->details->decrement('account_balance', $request->amount);
                }
                break;

            case 'credit':
                $user->details->increment('account_balance', $request->amount);
                break;

            default:
                break;
        }
        
        $alert_type = ucfirst($request->alert_type);
        $message = "You have received a {$request->alert_type} of {$transaction->total_amount} in your account.";

        if (request('send_email')) {
            $user->notify(new UserNotice(subject: "New {$alert_type} Transaction", message: $message));
        }
        return back()->with('success', $this->message('Action completed successfully.'));
    }


    public function changeTransferSettings(User $user) {
        if (request('validate_change')) {
            $user->details()->update([
                'permissions->validate_transfer' => true
            ]);
            return response()->json(['info' => 'Transfer validations enabled.']);
        } else {
            $user->details()->update([
                'permissions->validate_transfer' => false
            ]);
            return response()->json(['info' => 'Transfer validations disabled.']);
        }
    }

    private function createTransaction(User $user, Request $request,) {
        $amount = $user->details->currency.number_format($request->amount, 2);
        $transaction = TransactionLog::create([
            'user_id' => $user->id,
            'alert_type' => $request->alert_type == AlertType::Debit->value ? AlertType::Debit : AlertType::Credit,
            'transaction_type' => TransactionType::Local,
            'date' => $request->date,
            'description' => "Local Transfer of {$amount} to account.",
            'narration' => $request->note,
            'amount' => $request->amount,
            'currency' => $user->details->currency,
            'status' => 'successful',
            'active_code' => null,
            'txn_id' => txn_id(),
            'expires_at' => now(),
            'creator' => 'admin'
        ]);

        return $transaction;
    }
}
