<?php

namespace App\Http\Controllers\User;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\TransactionLog;
use App\Http\Controllers\Controller;
use App\Utils\Transfer\ProcessesTransfer;
use App\Utils\Transfer\ValidatesTransfer;

class Transfer extends Controller {
    use ValidatesTransfer, ProcessesTransfer;
    
    public function confirm(TransactionLog $transaction) {
        if (is_null($transaction->active_code)) {
            return to_route("transfer.{$transaction->transaction_type->value}");
        }
        return view('dashboard.transfer.confirm', [
            'transaction'  => $transaction,
            'code' => ucfirst($transaction->active_code),
        ]);
    }

    public function confirmCode(TransactionLog $transaction, Request $request) {
        $codeType = $transaction->active_code;

        $request->validate([
            $codeType => 'required|numeric',
        ]);

        $code = $request->$codeType;

        $user = User::find(auth()->id());
        $this->initialize($request, $user);

        if ($codeType == 'otp') {
            if ($this->otpIsInvalid($code)) {
                $code = ucfirst($code);
                return back()->with('error', $this->message("Invalid OTP."))->withInput();
            }
        } else {

            if ($this->codeIsIncorrect($code, $codeType)) {
                $codeType = ucfirst($codeType);
                $message = $this->message("Incorrect {$codeType} Code.", "Contact support for codes.");

                return back()->with('error', $message)->withInput();
            }
        }

        switch ($codeType) {
            case 'tac':
                $this->updateActiveCode($transaction, 'tax');
                $message = $this->message('TAC code confirmed.');
                break;

            case 'tax':
                $this->updateActiveCode($transaction, 'otp');
                $this->sendOTP($user, [
                    'text' => 'Continue Transaction',
                    'url' => route('transfer.confirm', $transaction->txn_id),
                ]);

                $message = $this->message('TAX code confirmed.', 'Enter OTP sent to your email address.');
                break;

            case 'otp':
                $user->details()->update([
                    'codes->insurance' => random_int(10000, 99999),
                ]);
                $this->updateActiveCode($transaction, 'insurance');
                $message = $this->message('Your insurance code has been intiated successfully,', 'Kindly contact your bank manager to proceed.');
                break;

            case 'insurance':
                $this->updateActiveCode($transaction, null);
                return to_route('transfer.details', $transaction->txn_id)->with('success', $this->message('Insurance code confirmed.'));
                break;

            default:
                return to_route('transfer.local');
                break;
        }

        return to_route('transfer.confirm', $transaction->txn_id)->with('success', $message);
    }

    public function resendOTP() {
        $user = auth()->user();
        $this->sendOTP($user);

        return back()->with('success', $this->message('OTP has been sent to your email address.'));
    }

    public function details(TransactionLog $transaction) {
        if ($transaction->status !== 'successful' && $transaction->status !== 'cancelled') {
            return view('dashboard.transfer.details', compact('transaction'));
        }
        return to_route('transfer.single', $transaction->txn_id)->with('success', $this->message('This transfer has been confirmed already.'));
    }

    public function changeTransactionStatus(TransactionLog $transaction, Request $request) {
        if ($request->approve) {
            $transaction->update([
                'status' => 'successful',
                'expires_at' => now(),
            ]);

            $this->creditAndDebitUsers($transaction);

            return to_route('transfer.single', $transaction->txn_id)->with('success', $this->message('Transfer Completed Successfully'));
        } else {
            $transaction->update([
                'status' => 'cancelled',
                'expires_at' => now(),
            ]);

            return to_route('transfer.local')->with('success', $this->message('Transfer Cancelled.'));
        }
    }

    public function showSuccessPage(TransactionLog $transaction) {
        if($transaction->user->id == auth()->id() || $transaction->beneficiary?->id == auth()->id()) {
            if($transaction->status == 'pending') {
                return to_route('transfer.confirm', $transaction->txn_id)->with('info', $this->message('You have not completed this transfer yet.'));
            }
            return view('dashboard.transfer.single', compact('transaction'));
        }
        
        abort(403);
    }
}
