<?php

namespace App\Http\Controllers\Admin;

use Exception;
use Throwable;
use App\Models\Setting;
use App\Mail\WelcomeEmail;
use App\Events\NotifyAdmin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class SiteSettingsController extends Controller {

    public function sendTestEmail(Request $request) {
        $request->validate([
            'email_receiver' => 'required',
        ]);

        try {
            Mail::to($request->email_receiver)->send(new WelcomeEmail(false, 'Successful Email Configuration'));
        } catch (Exception $e) {
            return response()->json([
                'error' => [
                    'title' => 'Error',
                    'message' => 'Wrong Email Configuration'
                ]
            ]);
        }
        return response()->json([
            'success' => [
                'title' => 'Success',
                'message' => 'Test Email Sent',
            ]
        ]);
    }
    public function displayConfigPage() {
        $settings = Setting::find(1);
        return view('admin.settings.config', [
            'settings' => $settings,
            'id' => 'emailTest',
            'title' => 'Test Configuration',
        ]);
    }

    public function updateConfig(Request $request) {
        $request->validate([
            'mail_driver' => 'required',
            'mail_host' => 'required',
            'mail_port' => 'required|numeric',
            'mail_from' => 'required',
            'mail_encryption' => 'nullable',
            'mail_username' => 'nullable|email',
            'mail_password' => 'nullable',
        ]);
        $settings = Setting::find(1);
        $settings->update([
            'mail_driver' => $request->mail_driver,
            'mail_host' => $request->mail_host,
            'mail_port' => $request->mail_port,
            'mail_from' => $request->mail_from,
            'mail_encryption' => $request->mail_encryption,
            'mail_username' => $request->mail_username,
            'mail_password' => $request->mail_password,
        ]);
        $settings = Setting::find(1);

        return response()->json([
            'success' => [
                'title' => 'Success',
                'message' => 'Site Config Updated.'
            ],
        ], 201);
    }

    public function displaySettingsPage() {
        $settings = Setting::find(1);
        return view('admin.settings.index', [
            'charges' => $settings->charges,
        ]);
    }

    public function updateSiteSettings() {
        $data = Validator::make(request()->all(), [
            'site_name' => 'required',
            'site_email' => 'required|email',
            'site_url' => 'required|url',
            'site_logo' => 'nullable|mimes:jpeg,jpg,png,gif,svg|max:3072',
            'site_favicon' => 'nullable|mimes:jpeg,jpg,png,gif,ico|max:3072',
            'site_alt_logo' => 'nullable|image|max:3072',
            'local_transfer_charge' => 'required|numeric|max:100',
            'international_transfer_charge' => 'required|numeric|max:100',
            'live_support' => 'nullable',
            'site_phone' => 'required',
            'location' => 'required'
        ]);
        $settings = Setting::find(1);
        if ($data->fails()) {
            return response()->json(['errors' => $data->errors()->all()]);
        } else {
            $data = $data->validated();

            $logo_path = $settings->site_logo;
            $favicon_path = $settings->site_favicon;
            $alt_logo_path = $settings->alt_site_logo;

            if (request()->hasFile('site_logo')) {
                $logo_path = request()->file('site_logo')->store('logos', 'storage');
                $logo_path = asset('/uploads/' . $logo_path);
            }
            if (request()->hasFile('site_favicon')) {
                $favicon_path = request()->file('site_favicon')->store('logos', 'storage');
                $favicon_path = asset('/uploads/' . $favicon_path);
            }
            if (request()->hasFile('site_alt_logo')) {
                $alt_logo_path = request()->file('site_alt_logo')->store('logos', 'storage');
                $alt_logo_path = asset('/uploads/' . $alt_logo_path);
            }
            $settings->update([
                'site_name' => $data['site_name'],
                'site_email' => $data['site_email'],
                'site_phone' => $data['site_phone'],
                'location' => $data['location'],
                'site_logo' => $logo_path,
                'site_favicon' => $favicon_path,
                'alt_site_logo' => $alt_logo_path,
                'charges' => [
                    "local" => $data["local_transfer_charge"],
                    "international" => $data["international_transfer_charge"],
                ],
                'site_url' => $data['site_url'],
                'live_support' => request('live_support'),
                'preferences' => [
                    'verify_email' => request('verify_email') ? true : false,
                ],
            ]);

            return response()->json(['success' => 'Settings Updated Successfully']);
        }
    }
}
